<?php

namespace App\Http\Controllers\Panel;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use App\DataTables\Backend\AirportDataTable;
use App\Models\Airport;
use App\Models\AirportHistory;
use App\Models\Country;
use App\Models\State;
use App\Models\City;
use Auth;

class AirportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(AirportDataTable $dataTable)
    {
        return $dataTable->render('panel.airports.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $countries = Country::pluck('name', 'id');

        return view('panel.airports.create', compact('countries'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->validate($request,[
            'name' => 'required',
            'iso_iata' => 'required|unique:airports',
            'country' => 'required',
            'state' => 'required',
            'city' => 'required',
        ]);

        $airport = new Airport();
        $airport->name = $request->name;
        $airport->iso_iata = strtoupper($request->iso_iata);
        $airport->country_id = $request->country;
        $airport->state_id = $request->state;
        $airport->city_id = $request->city;
        if ($airport->save()) {
            $history = new AirportHistory();
            $history->data = json_encode($_POST);
            $history->data_ip = ipGeo();
            $history->status = 'Creado';
            $history->airport_id = $airport->id;
            $history->user_id = Auth::user()->id;
            $history->save();

            return redirect()->route('airports.show', encrypt($airport->id))->with('success', 'Registro creado exitosamente.');
        } else {
            return redirect()->back()->with('error', 'Error al crear el registro.'); 
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $id = Crypt::decrypt($id);
        $airport = Airport::findOrFail($id);
        $history = $airport->airportHistory;

        return view('panel.airports.show', compact('airport', 'history'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $id = Crypt::decrypt($id);
        $airport = Airport::findOrFail($id);
        $countries = Country::pluck('name', 'id');
        $states = State::where('country_id', $airport->country_id)->pluck('name', 'id');
        $cities = City::where('state_id', $airport->state_id)->pluck('name', 'id');

        return view('panel.airports.edit', compact('airport', 'countries', 'states', 'cities'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $id = Crypt::decrypt($id);
        $airport = Airport::findOrFail($id);
        $this->validate($request,[
            'name' => 'required',
            'iso_iata' => 'required|unique:airports,iso_iata,'.$airport->id,
            'country' => 'required',
            'state' => 'required',
            'city' => 'required',
        ]);

        $airport->name = $request->name;
        $airport->iso_iata = strtoupper($request->iso_iata);
        $airport->country_id = $request->country;
        $airport->state_id = $request->state;
        $airport->city_id = $request->city;
        if ($airport->save()) {
            $history = new AirportHistory();
            $history->data = json_encode($_POST);
            $history->data_ip = ipGeo();
            $history->status = 'Modificado';
            $history->airport_id = $airport->id;
            $history->user_id = Auth::user()->id;
            $history->save();

            return redirect()->route('airports.show', encrypt($airport->id))->with('success', 'Registro actualizado exitosamente.');
        } else {
            return redirect()->back()->with('error', 'Error al crear el registro.'); 
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $id = Crypt::decrypt($id);
        $airport = Airport::findOrFail($id);
        
        $history = new AirportHistory();
        $history->data = json_encode($_POST);
        $history->data_ip = ipGeo();
        $history->status = 'Eliminado';
        $history->airport_id = $airport->id;
        $history->user_id = Auth::user()->id;
        $history->save();

        if ($airport->delete()) {
            return redirect()->route('airports.index')->with('success', '¡Registro eliminado exitosamente!');
        } else {
            return redirect()->back()->with('error', '¡Error!');
        }
    }
}
