<?php

namespace App\Http\Controllers\Panel;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use App\DataTables\Backend\UserDataTable;
use App\Models\User;
use App\Models\UserHistory;
use Spatie\Permission\Models\Role;
use Carbon\Carbon;
use Auth;


class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(UserDataTable $dataTable)
    {
        return $dataTable->render('panel.users.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = Role::pluck('name','id');

        return view('panel.users.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->validate($request,[
            'email' => 'required|email|unique:users',
            'role' => 'required',
            'password' => 'required',
            'name' => 'required',
            'last_name' => 'required',
        ]);

        $date = Carbon::now();
        $date = $date->format('Y-m-d h:m:s');

        $user = new User();
        $user->name = $request->name;
        $user->last_name = $request->last_name;
        $user->email = $request->email;
        $user->password = bcrypt($request->password);
        $user->type = 'Backend';
        $user->status = $request->status;
        $user->email_verified_at = $date;
        if ($user->save()) {
            $role = Role::findOrFail($request->role);
            $user->assignRole($role->id);

            $history = new UserHistory();
            $history->data = json_encode($_POST);
            $history->data_ip = ipGeo();
            $history->status = 'Creado';
            $history->user_edit = $user->id;
            $history->user_id = Auth::user()->id;
            $history->save();

            return redirect()->route('users.show', encrypt($user->id))->with('success', 'Registro creado exitosamente.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $id = Crypt::decrypt($id);
        $user = User::findOrFail($id);
        $history = UserHistory::where('user_edit', $user->id)->get();

        return view('panel.users.show', compact('user', 'history'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $id = Crypt::decrypt($id);
        $user = User::findOrFail($id);
        $roles = Role::pluck('name','id');
        $role = Role::where('name', implode( ", ", $user->getRoleNames()->toArray()))->first();

        return view('panel.users.edit', compact('user', 'roles', 'role'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $id = Crypt::decrypt($id);
        $user = User::findOrFail($id);

        $this->validate($request,[
            'email' => 'required|email|unique:users,email,'.$user->id,
            'role' => 'required',
            'status' => 'required',
            'name' => 'required',
            'last_name' => 'required',
        ]);
        
        $user->name = $request->name;
        $user->last_name = $request->last_name;
        $user->email = $request->email;
        if ($request->password) {
            $user->password = bcrypt($request->password);
        }
        $user->status = $request->status;
        if ($user->save()) {
            $role = Role::where('name', implode( ", ", $user->getRoleNames()->toArray()))->first();
            $user->removeRole($role->id);
            $user->assignRole($request->role);

            $history = new UserHistory();
            $history->data = json_encode($_POST);
            $history->data_ip = ipGeo();
            $history->status = 'Modificado';
            $history->user_edit = $user->id;
            $history->user_id = Auth::user()->id;
            $history->save();

            return redirect()->route('users.show', encrypt($user->id))->with('success', '¡Registro modificado exitosamente!');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
